#ifndef CE2_RSA_BUILD_H
#define CE2_RSA_BUILD_H

/*
* All the includes that are needed for code using this module to
* compile correctly should be #included here.
*/
#include "CE2_public.h"

#ifdef __cplusplus
extern "C"
{
#endif

/*
 *  Object %CE2_RSA_BUILD.h    : %
 *  State           :  %state%
 *  Creation date   :  %date%
 *  Last modified   :  %modify_time%
 */
/** @file
 *  \brief This module defines some utility functions for working with
 * RSA cryptography.
 *
 *  \version CE2_RSA_BUILD.h#1:incl:1
 *  \author ronys
 *  \remarks Copyright (C) 2001-2005 by Discretix Technologies Ltd. All Rights reserved.
 */

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Structs *****************************/
/************************ Public Variables ********************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name:
*  CE2_RSA_Build_PubKey
*
* Inputs:
*  @param UserPubKey_ptr [out] - A pointer to the public key structure. 
*        This structure is used as input to the CE2_RSA_PRIM_Encrypt API.
*  @param Exponent_ptr [in] - A pointer to the exponent stream of bytes (Big-Endian format)
*  @param ExponentSize [in] - The size of the exponent, in bytes.  
*  @param Modulus_ptr [in] - A pointer to the modulus stream of bytes (Big-Endian format).
*         The MS (most significant) bit must be set to '1'.
*  @param ModulusSize [in] - The size of the modulus, in bytes. 
*         Supported sizes are all 256 multiples between 512 - 2048.
*
* \brief \b
* Description:
*  CE2_RSA_Build_PubKey populates a CE2RSAPubKey_t structure with
*  the provided modulus and exponent.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PubKey
*     to build public key with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Build_PubKey(  CE2_RSAUserPubKey_t *UserPubKey_ptr,
                                    DxUint8_t *Exponent_ptr,
                                    DxUint16_t ExponentSize,
                                    DxUint8_t *Modulus_ptr,
                                    DxUint16_t ModulusSize );

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Build_PrivKey
*
* Inputs:
*  @param UserPrivKey_ptr [out] - A pointer to the public key structure.
*         This structure is used as input to the CE2_RSA_PRIM_Decrypt API.
*  @param PrivExponent_ptr [in] - A pointer to the private exponent stream of bytes;
*  @param PrivExponentSize [in] - The size of the private exponent, in bytes;
*  @param PubExponent_ptr  [in]- A pointer to the public exponent stream of bytes;
*  @param PubExponentSize  [in]- The size of the public exponent, in bytes;
*  @param Modulus_ptr      [in]- A pointer to the modulus stream of bytes.
*         The MS bit must be set to '1'.
*  @param ModulusSize      [in]- The size of the modulus, in bytes. 
*         Supported sizes are all 256 multiples between 512 - 2048.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Build_PrivKey populates a CE2RSAPrivKey_t structure with
*  the provided modulus and exponent, marking the key as a "non-CRT" key.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PrivKey
*     to build private key with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Build_PrivKey(CE2_RSAUserPrivKey_t   *UserPrivKey_ptr,
	                               DxUint8_t               *PrivExponent_ptr,
                                   DxUint16_t               PrivExponentSize,
                                   DxUint8_t               *PubExponent_ptr,
                                   DxUint16_t               PubExponentSize,
                                   DxUint8_t               *Modulus_ptr,
                                   DxUint16_t               ModulusSize );

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Build_PrivKeyCRT
*
* Inputs:
*  @param UserPrivKey_ptr [in] - a pointer to the publick key structure;
*  @param P_ptr [out] - a pointer to the first factor stream of bytes;
*  @param PSize [in/out] - the size of the first factor buffer in bytes, it is
*         updated to the actual size of the first factor, in bytes;
*  @param Q_ptr [out] - a pointer to the second factor stream of bytes;
*  @param QSize [in/out] - the size of the second factor buffer in bytes , it is
*         updated to the actual size of the second factor, in bytes;
*  @param dP_ptr [out] - a pointer to the first factors CRT exponent stream of bytes;
*  @param dPSize [in/out] - the size of the first factor exponent buffer in bytes, it is
*         updated to the actual size of the first factor exponent, in bytes;
*  @param dQ_ptr [out] - a pointer to the second factors CRT exponent stream of bytes;
*  @param dQSize [in/out] - the size of the second factors CRT exponent buffer in bytes, it is
*         updated to the actual size of the second factors CRT exponent, in bytes;
*  @param qInv_ptr [out] - a pointer to the first CRT coefficient stream of bytes;
*  @param qInvSize [in/out] - the size of the first CRT coefficient buffer in bytes, it is
*         updated to the actual size of the first CRT coefficient, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Build_PrivKeyCRT populates a CE2RSAPrivKey_t structure with
*  the provided parameters, marking the key as a "CRT" key.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PrivKeyCRT
*     to build private key with CRT values with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Build_PrivKeyCRT(CE2_RSAUserPrivKey_t *UserPrivKey_ptr,
                                      DxUint8_t *P_ptr, 
                                      DxUint16_t PSize,
                                      DxUint8_t *Q_ptr,
                                      DxUint16_t QSize,
                                      DxUint8_t *dP_ptr, 
                                      DxUint16_t dPSize,
                                      DxUint8_t *dQ_ptr,
                                      DxUint16_t dQSize,
                                      DxUint8_t *qInv_ptr,
                                      DxUint16_t qInvSize );

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Get_PubKey
*
* Inputs:
*  @param UserPubKey_ptr [in] - A pointer to the public key structure. 
*         This structure is used as input to the CE2_RSA_PRIM_Encrypt API.
*  @param Exponent_ptr   [out] - A pointer to the exponent stream of bytes
*  @param ExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes. 
*  @param Modulus_ptr    [in] - A pointer to the modulus stream of bytes.
*         The MS (most significant) bit must be set to '1'.
*  @param ModulusSize_ptr [in/out] - the size of the modulus buffer in bytes,
*         it is updated to the actual size of the modulus, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Get_PubKey gets the e,n public key from the database.
*  NOTE: All members of input UserPrivKey structure must be initialized,
*        including public key e pointer and it size.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PubKey
*     to get from public key public exponent (e) and modulus (N)
*     with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Get_PubKey(  CE2_RSAUserPubKey_t *UserPubKey_ptr,
                                  DxUint8_t  *Exponent_ptr,
                                  DxUint16_t *ExponentSize_ptr,
                                  DxUint8_t  *Modulus_ptr,
                                  DxUint16_t *ModulusSize_ptr ); 

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Get_ModSizeFromPubKey
*
* Inputs:
*  @param UserPubKey_ptr [in] - A pointer to the public key structure,
*         as returned by CE2_RSA_Build_PubKey.
*  @param ModulusSize_ptr [out] -  The actual size of the modulus, in bytes;
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*   CE2_RSA_Build_PubKey extracts the modulus size from a given
*   public key data structure.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Get_ModSizeFromPubKey
*     to get modulus size from public key with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Get_ModSizeFromPubKey(CE2_RSAUserPubKey_t *UserPubKey_ptr,
                                           DxUint16_t *ModulusSize_ptr );

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Get_PrivKey
*
* Inputs:
*  @param UserPrivKey_ptr [in] - A pointer to the private key structure;
*         This structure is used as input to the CE2_RSA_PRIM_Decrypt API.
*  @param PrivExponent_ptr [out] - A pointer to the exponent stream of bytes;
*  @param PrivExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes;
*  @param PubExponent_ptr [out] - a pointer to the public exponent stream of bytes;
*  @param PubExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes;
*  @param Modulus_ptr     [out] - A pointer to the modulus stream of bytes.
*         The MS (most significant) bit must be set to '1'.
*  @param ModulusSize_ptr [in/out] - the size of the modolous buffer in bytes,
*         it is updated to the actual size of the modolous, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Get_PrivKey gets the D, n private key from the database.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PubKey
*     to get from private key public exponent (e), private exponent
*     (d) and modulus (N) with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Get_PrivKey(CE2_RSAUserPrivKey_t *UserPrivKey_ptr,
				                   DxUint8_t             *PrivExponent_ptr,
				                   DxUint16_t            *PrivExponentSize_ptr,
				                   DxUint8_t             *PubExponent_ptr,
				                   DxUint16_t            *PubExponentSize_ptr,
				                   DxUint8_t             *Modulus_ptr,
				                   DxUint16_t            *ModulusSize_ptr );

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Get_PrivKeyCRT
*
* Inputs:
*  @param UserPrivKey_ptr [in] - a pointer to the publick key structure;
*  @param P_ptr [out] - a pointer to the first factor stream of bytes;
*  @param PSize_ptr [in/out] - the size of the first factor buffer in bytes, it is
*         updated to the actual size of the first factor, in bytes;
*  @param Q_ptr [out] - a pointer to the second factor stream of bytes;
*  @param QSize_ptr [in/out] - the size of the second factor buffer in bytes , it is
*         updated to the actual size of the second factor, in bytes;
*  @param dP_ptr [out] - a pointer to the first factors CRT exponent stream of bytes;
*  @param dPSize_ptr [in/out] - the size of the first factor exponent buffer in bytes, it is
*         updated to the actual size of the first factor exponent, in bytes;
*  @param dQ_ptr [out] - a pointer to the second factors CRT exponent stream of bytes;
*  @param dQSize_ptr [in/out] - the size of the second factors CRT exponent buffer in bytes, it is
*         updated to the actual size of the second factors CRT exponent, in bytes;
*  @param qInv_ptr [out] - a pointer to the first CRT coefficient stream of bytes;
*  @param qInvSize_ptr [in/out] - the size of the first CRT coefficient buffer in bytes, it is
*         updated to the actual size of the first CRT coefficient, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Get_PrivKeyCRT gets the p,q,dP,dQ,qP parts of private key
*  from the database.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PubKey
*     to get the p,q,dP,dQ,qP parts of private key from the database
*     with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Get_PrivKeyCRT(CE2_RSAUserPrivKey_t *UserPrivKey_ptr,
				                      DxUint8_t *P_ptr, 
				                      DxUint16_t *PSize_ptr,
				                      DxUint8_t *Q_ptr,
				                      DxUint16_t *QSize_ptr,
				                      DxUint8_t *dP_ptr, 
				                      DxUint16_t *dPSize_ptr,
				                      DxUint8_t *dQ_ptr,
				                      DxUint16_t *dQSize_ptr,
				                      DxUint8_t *qInv_ptr,
				                      DxUint16_t *qInvSize_ptr);

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Get_PrivKeyFull
*
* Inputs:
*  @param UserPrivKeyFull_ptr [in] - a pointer to the publick key structure;
*  @param PrivExponent_ptr [out] - A pointer to the exponent stream of bytes;
*  @param PrivExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes;
*  @param PubExponent_ptr [out] - a pointer to the public exponent stream of bytes;
*  @param PubExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes;
*  @param Modulus_ptr     [out] - A pointer to the modulus stream of bytes.
*         The MS (most significant) bit must be set to '1'.
*  @param ModulusSize_ptr [in/out] - the size of the modolous buffer in bytes,
*         it is updated to the actual size of the modolous, in bytes.
*  @param P_ptr [out] - a pointer to the first factor stream of bytes;
*  @param PSize_ptr [in/out] - the size of the first factor buffer in bytes, it is
*         updated to the actual size of the first factor, in bytes;
*  @param Q_ptr [out] - a pointer to the second factor stream of bytes;
*  @param QSize_ptr [in/out] - the size of the second factor buffer in bytes , it is
*         updated to the actual size of the second factor, in bytes;
*  @param dP_ptr [out] - a pointer to the first factors CRT exponent stream of bytes;
*  @param dPSize_ptr [in/out] - the size of the first factor exponent buffer in bytes, it is
*         updated to the actual size of the first factor exponent, in bytes;
*  @param dQ_ptr [out] - a pointer to the second factors CRT exponent stream of bytes;
*  @param dQSize_ptr [in/out] - the size of the second factors CRT exponent buffer in bytes, it is
*         updated to the actual size of the second factors CRT exponent, in bytes;
*  @param qInv_ptr [out] - a pointer to the first CRT coefficient stream of bytes;
*  @param qInvSize_ptr [in/out] - the size of the first CRT coefficient buffer in bytes, it is
*         updated to the actual size of the first CRT coefficient, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Get_PrivKeyFull gets the N,e,d,p,q,dP,dQ,qP parts of private key
*  from the database.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call CE2_RSA_Get_PrivKey and CE2_RSA_Get_PrivKeyCRT
*     to get the N,e,d,p,q,dP,dQ,qP parts of private key from the database
*     with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Get_PrivKeyFull(
        CE2_RSAUserPrivKey_t  *UserPrivKeyFull_ptr,
        DxUint8_t             *PrivExponent_ptr,
        DxUint16_t            *PrivExponentSize_ptr,
        DxUint8_t             *PubExponent_ptr,
        DxUint16_t            *PubExponentSize_ptr,
        DxUint8_t             *Modulus_ptr,
        DxUint16_t            *ModulusSize_ptr,
        DxUint8_t             *P_ptr, 
        DxUint16_t            *PSize_ptr,
        DxUint8_t             *Q_ptr,
        DxUint16_t            *QSize_ptr,
        DxUint8_t             *dP_ptr, 
        DxUint16_t            *dPSize_ptr,
        DxUint8_t             *dQ_ptr,
        DxUint16_t            *dQSize_ptr,
        DxUint8_t             *qInv_ptr,
        DxUint16_t            *qInvSize_ptr);


#ifdef __cplusplus
}
#endif

#endif /* CE2_RSA_BUILD_H */
